/*
 *      IP Virtual Server
 *      data structure and funcationality definitions
 */

#ifndef _IP_VS_H
#define _IP_VS_H

#define IP_VS_VERSION_CODE            0x000101
#define NVERSION(version)                       \
        (version >> 16) & 0xFF,                 \
        (version >> 8) & 0xFF,                  \
        version & 0xFF                       

/*
 *      Virtual Service Flags
 */
#define IP_VS_SVC_F_PERSISTENT        0x0001    /* persistent port */
#define IP_VS_SVC_F_HASHED            0x0002    /* hashed entry */

/*
 *      Destination Server Flags
 */
#define IP_VS_DEST_F_AVAILABLE        0x0001    /* Available tag */

/*
 * The default IP_VS_TEMPLATE_TIMEOUT is a little larger than average
 * connection time plus IPVS TCP FIN timeout(2*60*HZ). Because the
 * template won't be released until its controlled masq entries are
 * expired.
 * If IP_VS_TEMPLATE_TIMEOUT is too less, the template will soon expire
 * and will be put in expire again and again, which requires additional
 * overhead. If it is too large, the same will always visit the same
 * server, which will make dynamic load imbalance worse.
 */
#define IP_VS_TEMPLATE_TIMEOUT  6*60*HZ

#define IP_VS_SCHEDNAME_MAXLEN         16

struct ip_vs_rule_user {
        /* global options */
        int             tcp_timeout;    /* timeout values */
        int             tcp_fin_timeout;
        int             udp_timeout;
        
	/* virtual service options */
	u_int16_t	protocol;
	u_int32_t	vaddr;          /* virtual address */
	u_int16_t	vport;
        u_int32_t       vfwmark;        /* firwall mark of virtual */
        char            sched_name[IP_VS_SCHEDNAME_MAXLEN];
	unsigned	vs_flags;       /* virtual service flags */
	unsigned        timeout;        /* persistent timeout in ticks */
	u_int32_t	netmask;        /* persistent netmask */
        
	/* destination specific options */
	u_int32_t	daddr;          /* real destination address */
	u_int16_t	dport;
	unsigned        conn_flags;     /* destination flags */
	int             weight;         /* destination weight */
};


/*
 *      IPVS socket options
 */
#define IP_VS_BASE_CTL  	(64+1024+64)            /* base */

#define IP_VS_SO_SET_NONE	IP_VS_BASE_CTL	        /* just peek */
#define IP_VS_SO_SET_INSERT	(IP_VS_BASE_CTL+1)
#define IP_VS_SO_SET_ADD	(IP_VS_BASE_CTL+2)
#define IP_VS_SO_SET_EDIT	(IP_VS_BASE_CTL+3)
#define IP_VS_SO_SET_DEL	(IP_VS_BASE_CTL+4)
#define IP_VS_SO_SET_FLUSH	(IP_VS_BASE_CTL+5)
#define IP_VS_SO_SET_LIST	(IP_VS_BASE_CTL+6)
#define IP_VS_SO_SET_ADDDEST	(IP_VS_BASE_CTL+7)
#define IP_VS_SO_SET_DELDEST	(IP_VS_BASE_CTL+8)
#define IP_VS_SO_SET_EDITDEST	(IP_VS_BASE_CTL+9)
#define IP_VS_SO_SET_TIMEOUTS	(IP_VS_BASE_CTL+10)
#define IP_VS_SO_SET_MAX	IP_VS_SO_SET_TIMEOUTS

#define IP_VS_SO_GET_VERSION	IP_VS_BASE_CTL
#define IP_VS_SO_GET_MAX	IP_VS_SO_GET_VERSION


#define IP_VS_CONN_F_FWD_MASK         0x0003    /* mask for the fwd methods */
#define IP_VS_CONN_F_MASQ	      0x0000    /* masquerading */
#define IP_VS_CONN_F_LOCALNODE	      0x0001    /* local node */
#define IP_VS_CONN_F_TUNNEL	      0x0002    /* tunneling */
#define IP_VS_CONN_F_DROUTE           0x0003    /* direct routing */
#define IP_VS_CONN_F_HASHED	      0x0040 	/* hashed entry */
#define IP_VS_CONN_F_NOOUTPUT         0x0080    /* no output packets */
#define IP_VS_CONN_F_INACTIVE         0x0100    /* not established */
#define IP_VS_CONN_F_OUT_SEQ          0x0200    /* must do output seq adjust */
#define IP_VS_CONN_F_IN_SEQ           0x0400    /* must do input seq adjust */
#define IP_VS_CONN_F_NO_CPORT         0x0800    /* no cport set yet */


#ifdef __KERNEL__

#include <linux/config.h>
#include <linux/list.h>                 /* for struct list_head */
#include <linux/spinlock.h>             /* for struct rwlock_t */
#include <linux/skbuff.h>               /* for struct sk_buff */
#include <linux/ip.h>                   /* for struct iphdr */
#include <asm/atomic.h>                 /* for struct atomic_t */


/*  #define CONFIG_IP_VS_DEBUG */

#ifdef CONFIG_IP_VS_DEBUG
extern int ip_vs_get_debug_level(void);
#define IP_VS_DBG(level, msg...)                        \
    do {                                                \
            if (level <= ip_vs_get_debug_level())       \
                    printk(KERN_DEBUG "IPVS: " ## msg); \
    } while (0)
#else	/* NO DEBUGGING at ALL */
#define IP_VS_DBG(level, msg...)  do {} while (0)
#endif

#define IP_VS_ERR(msg...) printk(KERN_ERR "IPVS: " ## msg )
#define IP_VS_INFO(msg...) printk(KERN_INFO "IPVS: " ## msg )
#define IP_VS_WARNING(msg...) \
	printk(KERN_WARNING "IPVS: " ## msg)

#ifdef CONFIG_IP_VS_DEBUG
#define EnterFunction(level)                                            \
    do {                                                                \
            if (level <= ip_vs_get_debug_level())                       \
                    printk(KERN_DEBUG "Enter: %s, %s line %i\n",        \
                           __FUNCTION__, __FILE__, __LINE__);           \
    } while (0)
#define LeaveFunction(level)                                            \
    do {                                                                \
            if (level <= ip_vs_get_debug_level())                       \
                        printk(KERN_DEBUG "Leave: %s, %s line %i\n",    \
                               __FUNCTION__, __FILE__, __LINE__);       \
    } while (0)
#else
#define EnterFunction(level)   do {} while (0)
#define LeaveFunction(level)   do {} while (0)
#endif

/*
 *  The port number of FTP service (in network order).
 */
#define FTPPORT  __constant_htons(21)
#define FTPDATA  __constant_htons(20)


/*
 *  IPVS sysctl variables under the /proc/sys/net/ipv4/vs/
 */
#define NET_IPV4_VS              21

enum {
	NET_IPV4_VS_DEBUG_LEVEL=1,
	NET_IPV4_VS_AMEMTHRESH=2,
	NET_IPV4_VS_AMDROPRATE=3,
	NET_IPV4_VS_DROP_ENTRY=4,
	NET_IPV4_VS_DROP_PACKET=5,
	NET_IPV4_VS_SECURE_TCP=6,
	NET_IPV4_VS_TO_ES=7,
	NET_IPV4_VS_TO_SS=8,
	NET_IPV4_VS_TO_SR=9,
	NET_IPV4_VS_TO_FW=10,
	NET_IPV4_VS_TO_TW=11,
	NET_IPV4_VS_TO_CL=12,
	NET_IPV4_VS_TO_CW=13,
	NET_IPV4_VS_TO_LA=14,
	NET_IPV4_VS_TO_LI=15,
	NET_IPV4_VS_TO_SA=16,
	NET_IPV4_VS_TO_UDP=17,
	NET_IPV4_VS_TO_ICMP=18,
	NET_IPV4_VS_LBLC_EXPIRE=19,
	NET_IPV4_VS_LBLCR_EXPIRE=19,
};

enum {
	IP_VS_S_NONE = 0,
	IP_VS_S_ESTABLISHED,
	IP_VS_S_SYN_SENT,
	IP_VS_S_SYN_RECV,
	IP_VS_S_SYNACK,
	IP_VS_S_FIN_WAIT,
	IP_VS_S_TIME_WAIT,
	IP_VS_S_CLOSE,
	IP_VS_S_CLOSE_WAIT,
	IP_VS_S_LAST_ACK,
	IP_VS_S_LISTEN,
	IP_VS_S_UDP,
	IP_VS_S_ICMP,
	IP_VS_S_LAST
};


struct ip_vs_timeout_table {
	atomic_t refcnt;
	int scale;
	int timeout[IP_VS_S_LAST+1];
};

union ip_vs_tphdr {
        unsigned char *raw;
        struct udphdr *uh;
        struct tcphdr *th;
        struct icmphdr *icmph;
        __u16 *portp;
};


/*
 *	Delta sequence info structure
 *	Each ip_vs_conn has 2 (output AND input seq. changes).
 *      Only used in the VS/NAT.
 */
struct ip_vs_seq {
        __u32           init_seq;       /* Add delta from this seq */
        __u32           delta;          /* Delta in sequence numbers */
        __u32           previous_delta; /* Delta in sequence numbers
                                           before last resized pkt */
};


/*
 *	IP_VS structure allocated for each dynamically scheduled connection
 */
struct ip_vs_conn {
	struct list_head        c_list;         /* hashed list heads */
	atomic_t                refcnt;         /* reference count */
	struct timer_list       timer;          /* Expiration timer */
	unsigned long           timeout;        /* timeout */
        __u32                   caddr;          /* client address */
        __u32                   vaddr;          /* virtual address */
        __u32                   daddr;          /* destination address */
        __u16                   cport;
        __u16                   vport;
        __u16                   dport;
	__u16                   protocol;       /* Which protocol (TCP/UDP) */
	struct ip_vs_conn       *control;       /* Master control connection */
	atomic_t                n_control;      /* Number of controlled ones */
	__u16                   flags;          /* status flags */
	__u16                   state;          /* state info */
	struct ip_vs_timeout_table *timeout_table;
	struct ip_vs_dest       *dest;          /* real server */
        atomic_t                in_pkts;        /* incoming packet counter */
        /* Note: we can group the following members into a structure,
           in order to save more space, and the following members are
           only used in VS/NAT anyway */
	struct ip_vs_app        *app;           /* bound ip_vs_app object */
	void                    *app_data;      /* Application private data */
        struct ip_vs_seq        in_seq;         /* incoming seq. struct */
        struct ip_vs_seq        out_seq;        /* outgoing seq. struct */
};


/*
 *	The information about the virtual service offered to the net
 *	and the forwarding entries
 */
struct ip_vs_service {
	struct list_head	s_list;   /* hashed d-linked list head */
	struct list_head	f_list;   /* hashed d-linked list head */
	atomic_t		refcnt;   /* reference counter */
	__u16			protocol; /* which protocol (TCP/UDP) */
	__u32			addr;	  /* IP address for virtual service */
	__u16			port;	  /* port number for the service */
	unsigned long		fwmark;   /* firewall mark of the service */
	unsigned		flags;	  /* service status flags */
	unsigned		timeout;  /* persistent timeout in ticks */
	__u32			netmask;  /* grouping granularity */
	struct list_head	destinations;  /* real server d-linked list */
	struct ip_vs_scheduler	*scheduler;    /* bound scheduler object */
	rwlock_t		sched_lock;    /* lock sched_data */
	void			*sched_data;   /* scheduler application data */
	atomic_t		packethandled; /* packet handled */
};


/*
 *	The real server destination forwarding entry
 *	with ip address, port
 */
struct ip_vs_dest {
	struct list_head	n_list;   /* d-linked list head */
	__u32			addr;	  /* IP address of real server */
	__u16			port;	  /* port number of the service */
	unsigned		flags;	  /* dest status flags */
	atomic_t		weight; 	/* server weight */
	atomic_t		conn_flags;	/* flags to copy to masq */
	atomic_t		activeconns;	/* active connections */
	atomic_t		inactconns;	/* inactive connections */
	atomic_t		refcnt; 	/* reference counter */
	struct list_head	d_list;   /* table with all dests */

	/* for virtual service */
	__u16			protocol; /* which protocol (TCP/UDP) */
	__u32			vaddr;	  /* IP address for virtual service */
	__u16			vport;	  /* port number for the service */
	unsigned long		vfwmark;  /* firewall mark of the service */
};


/*
 *	The scheduler object
 */
struct ip_vs_scheduler {
	struct list_head        n_list;   /* d-linked list head */
	char 			*name;    /* scheduler name */
	atomic_t		refcnt;   /* reference counter */
	struct module		*module;  /* THIS_MODULE/NULL */

        /* scheduler initializing service */
	int (*init_service)(struct ip_vs_service *svc);
        /* scheduling service finish */
        int (*done_service)(struct ip_vs_service *svc);
        /* scheduler updating service */
        int (*update_service)(struct ip_vs_service *svc);

	/* selecting a server from the given service */
	struct ip_vs_dest* (*schedule)(struct ip_vs_service *svc,
                                       struct iphdr *iph);
};


/*
 *	The application module object
 */
struct ip_vs_app
{
        struct list_head        n_list;   /* d-linked list head */
        char                    *name;    /* name of application module */
        unsigned                type;     /* type = proto<<16 | port
                                             (host byte order)*/
        /*  atomic_t                refcnt; */   /* reference counter */
	struct module		*module;  /* THIS_MODULE/NULL */
        
        /* ip_vs_app initializer */
        int (*init_conn)(struct ip_vs_app *, struct ip_vs_conn *);
        /* ip_vs_app finish */
        int (*done_conn)(struct ip_vs_app *, struct ip_vs_conn *);
        /* output hook */
        int (*pkt_out)(struct ip_vs_app *,
                       struct ip_vs_conn *, struct sk_buff **);
        /* input hook */
        int (*pkt_in)(struct ip_vs_app *,
                      struct ip_vs_conn *, struct sk_buff **);
};


/*
 *      IPVS core functions
 *      (from ip_vs_core.c)
 */
extern struct ip_vs_conn * ip_vs_schedule(struct ip_vs_service *svc,
                                          struct iphdr *iph);
const char *vs_proto_name(unsigned proto);


/*
 *     ip_vs_conn handling functions
 *     (from ip_vs_conn.c)
 */

/*
 *     IPVS connection entry hash table
 */
#ifndef CONFIG_IP_VS_TAB_BITS
#define CONFIG_IP_VS_TAB_BITS   12
#endif
#define IP_VS_CONN_TAB_BITS     CONFIG_IP_VS_TAB_BITS
#define IP_VS_CONN_TAB_SIZE     (1 << IP_VS_CONN_TAB_BITS)
#define IP_VS_CONN_TAB_MASK     (IP_VS_CONN_TAB_SIZE - 1)

#define VS_STATE_INPUT	        0
#define VS_STATE_OUTPUT	        4
#define VS_STATE_INPUT_ONLY	8

extern struct ip_vs_timeout_table vs_timeout_table;
extern struct ip_vs_timeout_table vs_timeout_table_dos;
extern atomic_t ip_vs_conn_no_cport_cnt;

extern int ip_vs_conn_hash(struct ip_vs_conn *cp);
extern int ip_vs_conn_unhash(struct ip_vs_conn *cp);
extern struct ip_vs_conn *ip_vs_conn_in_get
(int protocol, __u32 s_addr, __u16 s_port, __u32 d_addr, __u16 d_port);
extern struct ip_vs_conn *ip_vs_conn_out_get
(int protocol, __u32 s_addr, __u16 s_port, __u32 d_addr, __u16 d_port);
extern void ip_vs_conn_put(struct ip_vs_conn *cp);
extern struct ip_vs_conn *
ip_vs_conn_new(int proto, __u32 maddr, __u16 mport, __u32 saddr,
               __u16 sport, __u32 daddr, __u16 dport, unsigned flags);
extern const char * ip_vs_state_name(int state);
extern int vs_set_state(struct ip_vs_conn *cp, int state_off,
                        struct iphdr *iph, void *tp);
extern int ip_vs_conn_listen(struct ip_vs_conn *cp);
extern int ip_vs_check_template(struct ip_vs_conn *ct);
extern void ip_vs_secure_tcp_set(int on);
extern void ip_vs_set_state(struct ip_vs_conn *cp, int new_state);
extern void ip_vs_bind_dest(struct ip_vs_conn *cp, struct ip_vs_dest *dest);
extern void ip_vs_unbind_dest(struct ip_vs_conn *cp);
extern void ip_vs_random_dropentry(void);
extern int ip_vs_conn_init(void);
extern void ip_vs_conn_cleanup(void);

static inline void __ip_vs_conn_put(struct ip_vs_conn *cp) 
{
	atomic_dec(&cp->refcnt);
}

static inline void ip_vs_control_del(struct ip_vs_conn *cp)
{
	struct ip_vs_conn *ctl_cp = cp->control;
	if (!ctl_cp) {
		IP_VS_ERR("request control DEL for uncontrolled: "
                          "%d.%d.%d.%d:%d to %d.%d.%d.%d:%d\n",
                          NIPQUAD(cp->caddr),ntohs(cp->cport),
                          NIPQUAD(cp->vaddr),ntohs(cp->vport));
                return;
	}
        
	IP_VS_DBG(1, "DELeting control for: "
                  "cp.dst=%d.%d.%d.%d:%d ctl_cp.dst=%d.%d.%d.%d:%d\n",
                  NIPQUAD(cp->caddr),ntohs(cp->cport),
                  NIPQUAD(ctl_cp->caddr),ntohs(ctl_cp->cport));
        
	cp->control = NULL;
	if (atomic_read(&ctl_cp->n_control) == 0) {
		IP_VS_ERR("BUG control DEL with n=0 : "
                          "%d.%d.%d.%d:%d to %d.%d.%d.%d:%d\n",
                          NIPQUAD(cp->caddr),ntohs(cp->cport),
                          NIPQUAD(cp->vaddr),ntohs(cp->vport));
                return;
	}
	atomic_dec(&ctl_cp->n_control);
}

static inline void
ip_vs_control_add(struct ip_vs_conn *cp, struct ip_vs_conn *ctl_cp)
{
	if (cp->control) {
		IP_VS_ERR("request control ADD for already controlled: "
                          "%d.%d.%d.%d:%d to %d.%d.%d.%d:%d\n",
                          NIPQUAD(cp->caddr),ntohs(cp->cport),
                          NIPQUAD(cp->vaddr),ntohs(cp->vport));
		ip_vs_control_del(cp);
	}
        
	IP_VS_DBG(1, "ADDing control for: "
                  "cp.dst=%d.%d.%d.%d:%d ctl_cp.dst=%d.%d.%d.%d:%d\n",
                  NIPQUAD(cp->caddr),ntohs(cp->cport),
                  NIPQUAD(ctl_cp->caddr),ntohs(ctl_cp->cport));
        
	cp->control = ctl_cp;
	atomic_inc(&ctl_cp->n_control);
}



/*
 *      IPVS application functions
 *      (from ip_vs_app.c)
 */
#define IP_VS_APP_MAX_PORTS  8
extern int register_ip_vs_app(struct ip_vs_app *mapp,
                              unsigned short proto, __u16 port);
extern int unregister_ip_vs_app(struct ip_vs_app *mapp);
extern struct ip_vs_app * ip_vs_bind_app(struct ip_vs_conn *cp);
extern int ip_vs_unbind_app(struct ip_vs_conn *cp);
extern int ip_vs_app_pkt_out(struct ip_vs_conn *, struct sk_buff **skb_p);
extern int ip_vs_app_pkt_in(struct ip_vs_conn *, struct sk_buff **skb_p);
extern struct sk_buff * ip_vs_skb_replace (struct sk_buff *skb, int pri,
                                           char *o_buf, int o_len,
                                           char *n_buf, int n_len);
extern int ip_vs_app_init(void);
extern void ip_vs_app_cleanup(void);


/*
 *      Registering/unregistering scheduler functions
 *      (from ip_vs_sched.c)
 */
extern int register_ip_vs_scheduler(struct ip_vs_scheduler *scheduler);
extern int unregister_ip_vs_scheduler(struct ip_vs_scheduler *scheduler);
extern int ip_vs_register_scheduler_module(struct ip_vs_scheduler *scheduler);
extern void ip_vs_unregister_scheduler_module(struct ip_vs_scheduler *scheduler);
extern int ip_vs_bind_scheduler(struct ip_vs_service *svc,
                                struct ip_vs_scheduler *scheduler);
extern int ip_vs_unbind_scheduler(struct ip_vs_service *svc);
extern struct ip_vs_scheduler *ip_vs_lookup_scheduler(const char *sched_name);


/*
 *      IPVS control data and functions
 *      (from ip_vs_ctl.c)
 */
extern rwlock_t __ip_vs_svc_lock;
extern atomic_t ip_vs_dropentry;
extern struct ip_vs_service *
ip_vs_lookup_service(unsigned long fwmark,
                     __u16 protocol, __u32 vaddr, __u16 vport);
extern struct ip_vs_service * ip_vs_lookup_svc_fwm(unsigned long fwmark);
extern struct ip_vs_dest *
ip_vs_lookup_real_service(__u16 protocol, __u32 daddr, __u16 dport);
extern void update_defense_level(void);
extern void ip_vs_random_dropentry(void);
extern int ip_vs_control_init(void);
extern void ip_vs_control_cleanup(void);


/*
 *	This is a simple mechanism to ignore packets when
 *	we are loaded. Just set ip_vs_drop_rate to 'n' and
 *	we start to drop 1/rate of the packets
 */
extern int ip_vs_drop_rate;
extern int ip_vs_drop_counter;

static __inline__ int ip_vs_todrop(void)
{
	if (!ip_vs_drop_rate) return 0;
	if (--ip_vs_drop_counter > 0) return 0;
	ip_vs_drop_counter = ip_vs_drop_rate;
	return 1;
}


/*
 *      Slow timer functions for IPVS
 *      (from ip_vs_timer.c)
 */
extern void add_sltimer(struct timer_list * timer);
extern int  del_sltimer(struct timer_list * timer);
extern void mod_sltimer(struct timer_list *timer, unsigned long expires);
extern void ip_vs_sltimer_init(void);
extern void ip_vs_sltimer_cleanup(void);


/*
 *      ip_vs_fwd_tag returns the forwarding tag of the connection
 */
#define IP_VS_FWD_METHOD(cp)  (cp->flags & IP_VS_CONN_F_FWD_MASK)

extern __inline__ char ip_vs_fwd_tag(struct ip_vs_conn *cp)
{
        char fwd = 'M';

        switch (IP_VS_FWD_METHOD(cp)) {
        case IP_VS_CONN_F_LOCALNODE:
                fwd = 'L'; break;
        case IP_VS_CONN_F_TUNNEL:
                fwd = 'T'; break;
        case IP_VS_CONN_F_DROUTE:
                fwd = 'R'; break;
        }
        return fwd;
}


#endif /* __KERNEL__ */

#endif	/* _IP_VS_H */
